using System;
using System.IO;
using System.Text;
using Ravioli.ArchiveInterface;

namespace Ravioli.SamplePlugin
{
	public class Wad3File : ArchiveBase
	{
		private Wad3DirectoryEntry[] directory;

		public override string TypeName
		{
			get { return "Quake WAD (WAD2/WAD3) File (Sample)"; }
		}

		public override string[] Extensions
		{
			get { return new string[] { ".wad" }; }
		}

		public override IFileInfo[] Files
		{
			get
			{
				ArchiveFileInfo[] array = new ArchiveFileInfo[this.directory.Length];
				for (int i = 0; i < this.directory.Length; i++)
				{
					Wad3DirectoryEntry entry = this.directory[i];
					array[i] = new ArchiveFileInfo(i, entry.Name, entry.CompressedSize);
				}
				return array;
			}
		}

		protected override void ExtractFile(IFileInfo file, Stream stream, string outputFileName)
		{
			Wad3DirectoryEntry entry = this.directory[file.ID];
			FileStream outputStream = new FileStream(outputFileName, FileMode.Create, FileAccess.Write);
			try
			{
				CopyData(entry, stream, outputStream, -1);
				outputStream.Close();
			}
			catch (Exception)
			{
				outputStream.Close();
				if (File.Exists(outputFileName))
					File.Delete(outputFileName);
				throw;
			}
		}

		protected override void ExtractFile(IFileInfo file, Stream stream, Stream outputStream)
		{
			CopyData(this.directory[file.ID], stream, outputStream, -1);
		}

		protected override void ExtractFile(IFileInfo file, Stream stream, Stream outputStream, long byteCount)
		{
			CopyData(this.directory[file.ID], stream, outputStream, byteCount);
		}

		protected override bool IsValidFormat(Stream stream, BinaryReader reader)
		{
			stream.Seek(0, SeekOrigin.Begin);
			uint signature = reader.ReadUInt32();
			return (signature == 0x32444157 || signature == 0x33444157); // WAD2/WAD3 signature
		}

		protected override void ReadDirectory(Stream stream, BinaryReader reader)
		{
			if (!IsValidFormat(stream, reader))
				throw new InvalidDataException("This is not a valid WAD file.");

			// Header
			stream.Seek(0, SeekOrigin.Begin);
			uint signature = reader.ReadUInt32();
			uint fileCount = reader.ReadUInt32();
			uint directoryOffset = reader.ReadUInt32();

			// Directory
			stream.Seek(directoryOffset, SeekOrigin.Begin);
			Wad3DirectoryEntry[] newDirectory = new Wad3DirectoryEntry[fileCount];
			for (uint i = 0; i < fileCount; i++)
			{
				Wad3DirectoryEntry entry = new Wad3DirectoryEntry();
				entry.Offset = reader.ReadUInt32();
				entry.CompressedSize = reader.ReadUInt32();
				entry.UncompressedSize = reader.ReadUInt32();
				entry.Type = reader.ReadByte();
				entry.CompressionType = reader.ReadByte();
				entry.Padding = reader.ReadUInt16();
				byte[] nameBytes = reader.ReadBytes(16);
				entry.Name = Encoding.GetEncoding(1252).GetString(nameBytes);
				if (entry.Name.Contains("\0"))
				{
					int zeroPos = entry.Name.IndexOf('\0');
					entry.Name = entry.Name.Substring(0, zeroPos);
				}
				newDirectory[i] = entry;
			}
			this.directory = newDirectory;
		}

		protected override void OnClose()
		{
			this.directory = null;
		}

		private void CopyData(Wad3DirectoryEntry entry, Stream stream, Stream outputStream, long byteCount)
		{
			stream.Seek(entry.Offset, SeekOrigin.Begin);
			long remaining = (byteCount >= 0 && byteCount < entry.CompressedSize) ? byteCount : entry.CompressedSize;
			const int bufferSize = 8192;
			byte[] buffer = new byte[bufferSize];
			while (remaining > 0)
			{
				int bytesToRead = (remaining > bufferSize ? bufferSize : (int)remaining);
				int bytesRead = stream.Read(buffer, 0, bytesToRead);
				outputStream.Write(buffer, 0, bytesRead);
				remaining -= (uint)bytesRead;
			}
		}
	}
}
